#include <stdio.h>
#include <math.h>
#include <time.h>

#include "paldecode.h"

/*
			Color stuff
*/
#define Pi			3.14159265358979323846

#define lsize		2268	// raw capture pitch
#define rawwidth	2268	// raw capture scanline length

#define outwidth	360		// output field width
#define outheight	280		// output field height


static bool fast_color = true;
static int brightness = 0;

static short cosint[1024];		// cosinus table
static unsigned char bufY[2][outwidth];
static char bufU[outwidth / 2], bufV[outwidth / 2];
static int prevU[outwidth / 2], prevV[outwidth / 2];
static unsigned char clipb[1024];


static int  colorburst;			// colorburst phase offset
static bool oddframe;			// colorburst frame (odd/even)
static int  FcHF;				// horizontal color phase stepsize (.32)
static int  FcVF;				// vertical color phase stepsize (.32)

#define FcHF0 (229.2516 * 0x100000000L /lsize)
#define FcVF0 (-0x40000000L)

static long show_settings_timeout;

void PalColorDecoder::processKey(WPARAM key)
{
	bool show_it = false;

	switch(key)
   {
   	case VK_PRIOR:
      	show_it = true;
	      if (brightness < 255)
	         brightness++;
         break;
      case VK_NEXT:
      	show_it = true;
         if (brightness > -255)
	         brightness--;
         break;

   	case 'C':
      	fast_color = !fast_color;
         break;
	}

   if (show_it)
   	show_settings_timeout = time(0) + 5;
}

const char* PalColorDecoder::getHelp(void) const
{
	return("  PG UP/DOWN: Adjust brightness\n"
   		 "  C: Toggle fast color mode\n");
}

/*
 PAL software decoder
 ( including Y/C separation and YCrCb to RGB conversion )


INTRODUCTION TO PAL

 PAL is short for "Phase Alternating Line". The PAL system is a downward
 compatible video standard, which allows the possibility of viewing PAL
 color transmissions on older B/W televion sets.

 The PAL system consists of a single composite signal. The luminance and
 chrominance are combined (added) to build this signal. The luminance
 (brightness) is the average level of the signal, the chrominance (color)
 signals are the small patterns (waves).


DECODING THE SIGNAL

 The 2 chrominance components (U and V) are modulated on a sub-carier with
 a frequency of 4433618.75Hz. This is (1135/4 + 1/625) times the line
 frequency (Fl). The phase difference of U and V is always 90. This means
 U is modulated with the sine and V with the cosine of the colorburst
 phase.

 Uc = U * sin( Fc * 2*Pi )
 Vc = V * cos( Fc * 2*Pi )

 The phase of the modulated signals Uc and Vc is shifted 90 every line.
 On top of this, the Vc signal is inverted every other line. This feature
 of PAL is great for cancelling phase-errors, which effect the "hue" of
 colors. A change in hue is much more visible then a change in saturation.

 Now the three components Y, Uc and Vc are combined through addition :

  Ps = Y + Uc + Vc

 To decode the PAL video format you'd probably first seperate the luma and
 chrominance components (Y/C). To compute "Y" you'll have to remove the
 color frequency from the input signal. To compute "C" you extract it from
 the input signal. The Uc and Vc are seperated by phase (multiply with sine
 and cosine filters) and Vc is inverted (back) every other line.

 Two lines of chrominance samples are combined to cancel phase errors. The
 phase offset is determined by examining the phase of the "colorburst",
 which is located in every scanline just before the visible region. The
 phase of this colorburst is the phase of Vc.

 Finally, the equations from YUV back to RGB mode are :

  Red   = Y                +  1.140 * V
  Blue  = Y  +  2.028 * U
  Green = Y  -  0.391 * U  -  0.581 * V


ABOUT THIS DECODER

 Usually you would first seperate Y/C, then split the U and V channels and
 finally filter the data into pixels. By using only 2 filters I have combined
 all these steps to 1 pass filtering using 2 filters (cosine and sine) which
 are then combined according to the colorburst phase to build U and V samples.


 author :  Ewald Snel, 1998 - 1999.
*/


void initColorDecoder(int color_decoder_offset)
{
	for (int i=0; i < 1024; i++)
	{
		cosint[i] = (int)(0x7fffffff * cos( i*Pi /512.0 )) >> 16;

      int v = i - 320 + color_decoder_offset;
      if (v < 0)
      	v = 0;
      else if (v > 255)
      	v = 255;
		clipb[i] = v;
	}
}


static void detectPalCB(unsigned char* rawdata)
{
	static int cbPhase[288];				// colorburst phase offset
	double invPi = 0x80000000L /Pi;
	double amp;
	int fc0, fc1, fs0, fs1, p0, p1, p2, p3;
	int i,x,y;

	// Calculate some phase offset, linear interpolate the rest
	for (y=0; y < 288; y++)
	{
		int fcos=0, fsin=0, rs;
		unsigned char *rsdata = rawdata + rawwidth*y;//200
		unsigned int fss=0;

		for (x=0; x < 72; x++)
		{
			rs    = *rsdata++;
			fcos += cosint[fss >> 22] * rs;
			fsin += cosint[(fss - 0x40000000) >> 22] * rs;
			fss  += (unsigned int)FcHF0;
		}
   	// FP code
      double mydiv = sqrt( (double) fcos*fcos + (double) fsin*fsin );
      if (mydiv != 0)
			amp  = 1 / mydiv;
      else
      	amp = 10000;

		fc1 = (unsigned int) (acos( fcos * amp ) * invPi);
		fs0 = (unsigned int) (asin( fsin * amp ) * invPi)  +  0x80000000;

		// determine phase offset of colorburst
		fc0 = -fc1;
		fs1 = 0x80000000 - fs0;

		p0 = abs(fc0 - fs0);  p1 = abs(fc0 - fs1);
		p2 = abs(fc1 - fs0);  p3 = abs(fc1 - fs1);

		if (p0<p2 && p0<p3 || p1<p2 && p1<p3)  fc1 = fc0;
		if (p0<p1 && p0<p3 || p2<p1 && p2<p3)  fs1 = fs0;

		cbPhase[y] = (fs1 >> 1) + (fc1 >> 1);
	}

	// Vc polarity detection  ( even or odd frame )
	{
		int isr=0, isn=0;

		for (i=2; i < 286; i += 2)
		{
			isr += abs( cbPhase[i] - cbPhase[i - 1] ) >> 8;
			isn += abs( cbPhase[i + 1] - cbPhase[i] ) >> 8;
		}
		oddframe = (isr < isn);
	}

	// Sampling rate correction detection  ( vertical colorburst phase shift )
	{
		int vr=0;

		for (i=8; i < 144; i++)
		{
			vr += (cbPhase[144 + i] - cbPhase[i]) >> 8;
		}
		vr			= ((vr /144) << 8) /136;
		FcHF		= (int)(vr /lsize   +  FcHF0);
		FcVF		= vr          +  FcVF0;
		colorburst	= cbPhase[0]  +  200 * FcHF  -  0x60000000L;
		colorburst -= (oddframe ? 0 : -0x40000000);
	}
}

/*
 Decode single scanline of RAW samples in PAL format
 grayscale (Y) samples at half PAL resolution ...
*/
static void decodePalScanL(unsigned char* src, bool invInput )
{
	register int l, x;
   int lumo;

	for (int y=0; y < 2; y++)
	{
		lumo=0;

		for (x=0; x < 32; x++)
			lumo += src[x-112];

		lumo = 280 - (lumo >> 2);

		for (x=0; x < outwidth; x++)
		{
		// luma notchfilter, remove color frequency, downsample
			l  = lumo;
			l +=     3 * (src[ 0] + src[1]);
			l +=    -    (src[-1] + src[2]);
			l +=     2 * (src[-3] + src[4]);
			l +=         (src[-4] + src[5]);
			l +=    -    (src[-5] + src[6]);

			if (invInput)
			{
				l = ((700 - l)*3) >> 3;
			}
			else
			{
				l >>= 1;
			}

			// store luma sample, increase counter
			bufY[y][x] = ((l < 0) ? 0 : ((l > 255) ? 255 : l));

			src += 5;
		}
		src += rawwidth - 5*outwidth;
	}
}

static void decodePalScanUVSlow(unsigned char* src, int Fc, int lFcHF, int lFcVF, bool invCrV)
{
	int v = 0, u = 0;

	for (int x=0; x < outwidth/2; x++)
	{
		for (int sy=0; sy < 2; sy++)
		{
      	int c,s;
         int fsinc, fcosc;
		for (int sx=0; sx < 2; sx++)
		{
         // cosine filter samples, extract color frequency, downsample
         c  = (2204 * (src[ 0] - src[1])
			   +  2603 * (src[-1] - src[2])
			   -   903 * (src[-2] - src[3])
			   -  1500 * (src[-3] - src[4])
			   +   140 * (src[-4] - src[5])
			   +   339 * (src[-5] - src[6])) >> 4;

			// sine filter samples, extract color frequency, downsample
			s  = (1476 * (src[ 0] + src[1])
			   -   514 * (src[-1] + src[2])
			   -  1219 * (src[-2] + src[3])
			   -   223 * (src[-3] + src[4])
			   +   339 * (src[-4] + src[5])
			   +   140 * (src[-5] + src[6])) >> 4;

			// combine sine and cosine data at color frequency phase
			fsinc = cosint[(Fc - 0x40000000) >> 22];
			fcosc = cosint[Fc >> 22];

			// combine multiple (2x2) samples to minimize phase variance
			u += (fsinc*c + fcosc*s) >> 2;
			v += (fcosc*c - fsinc*s) >> 2;

			Fc  += lFcHF;
			src += 5;
		}
			v = -v;
			src += rawwidth - 10;
			Fc  += lFcVF - 2*lFcHF;
		}

		if (invCrV)
			v = -v;

		src += 10 - 2*rawwidth;
		Fc  += 2*lFcHF - 2*lFcVF;

		int lu = prevU[x];
		int lv = prevV[x];
		prevU[x] = u;
		prevV[x] = v;

		// combine 2 lines to correct phase variance
		u = (u + lu) >> 20;
		v = (v + lv) >> 20;

		// store chroma u,v samples
		bufU[x] = ((u < -128) ? -128 : ((u > 127) ? 127 : u));
		bufV[x] = ((v < -128) ? -128 : ((v > 127) ? 127 : v));
	}
}


static void decodePalScanUV(unsigned char* src, int Fc, int lFcHF, int lFcVF, bool invCrV)
{
	int v = 0, u = 0;

	for (int x=0; x < outwidth/2; x++)
	{
		for (int sy=0; sy < 2; sy++)
		{
      	int c,s;
         int fsinc, fcosc;

         // cosine filter samples, extract color frequency, downsample
         c  = (2204 * (src[ 0] - src[1])
			   +  2603 * (src[-1] - src[2])
			   -   903 * (src[-2] - src[3])
			   -  1500 * (src[-3] - src[4])
			   +   140 * (src[-4] - src[5])
			   +   339 * (src[-5] - src[6])) >> 4;

			// sine filter samples, extract color frequency, downsample
			s  = (1476 * (src[ 0] + src[1])
			   -   514 * (src[-1] + src[2])
			   -  1219 * (src[-2] + src[3])
			   -   223 * (src[-3] + src[4])
			   +   339 * (src[-4] + src[5])
			   +   140 * (src[-5] + src[6])) >> 4;

			// combine sine and cosine data at color frequency phase
			fsinc = cosint[(Fc - 0x40000000) >> 22];
			fcosc = cosint[Fc >> 22];

			// combine multiple (2) samples to minimize phase variance
			u += (fsinc*c + fcosc*s) >> 1;
			v += (fcosc*c - fsinc*s) >> 1;

			v = -v;
			src += rawwidth;
			Fc  += lFcVF;
		}

		if (invCrV)
			v = -v;

		src += 10 - 2*rawwidth;
		Fc  += 2*lFcHF - 2*lFcVF;

		int lu = prevU[x];
		int lv = prevV[x];
		prevU[x] = u;
		prevV[x] = v;

		// combine 2 lines to correct phase variance
		u = (u + lu) >> 20;
		v = (v + lv) >> 20;

		// store chroma u,v samples
		bufU[x] = ((u < -128) ? -128 : ((u > 127) ? 127 : u));
		bufV[x] = ((v < -128) ? -128 : ((v > 127) ? 127 : v));
	}
}


void PalColorDecoder::decodeFrame(unsigned char *src, int offset, bool invert_polarity)
{
	unsigned int Fc;
   RGBTRIPLE pixels1[outwidth];
   RGBTRIPLE pixels2[outwidth];

   src += offset;

   static bool init = false;
   if (!init)
   {
   	initColorDecoder(0);
      init = true;
   }

	// detect colorburst
	detectPalCB(src);

   src += getColorBurstWidth();

   if (!_color_mode)
   {
   	memset(bufU, 0, outwidth/2);
   	memset(bufV, 0, outwidth/2);
   }

	Fc = colorburst - FcHF - (FcHF >> 1);

   lockDisplay();

   if (_color_mode)
   {
      // prevV en prevU ininitaliseren voor de 1ste lijn ...
	  	decodePalScanUV(src+2*rawwidth, Fc+2*FcVF, 5*FcHF, FcVF, oddframe);
   }

	for (int y=0; y < outheight; y+=2)
	{
		decodePalScanL( src, invert_polarity );
      if (_color_mode)
      {
      	if (fast_color)
		  		decodePalScanUV( src, Fc, 5*FcHF, FcVF, oddframe);
         else
            decodePalScanUVSlow( src, Fc, 5*FcHF, FcVF, oddframe);
      }
		// store pixels
		for (int x=0; x < outwidth; x+=2)
		{
			int u = bufU[x>>1];
			int v = bufV[x>>1];
			int cr = (64*320+32 +         73*v) >> 6;
			int cg = (64*320+32 -  25*u - 37*v) >> 6;
			int cb = (64*320+32 + 130*u       ) >> 6;
         int Y;

			Y = bufY[0][x];
			pixels1[x].rgbtRed = clipb[Y + cr];
         pixels1[x].rgbtGreen = clipb[Y + cg];
         pixels1[x].rgbtBlue = clipb[Y + cb];

			Y = bufY[0][x+1];
			pixels1[x+1].rgbtRed = clipb[Y + cr];
         pixels1[x+1].rgbtGreen = clipb[Y + cg];
         pixels1[x+1].rgbtBlue = clipb[Y + cb];

			Y = bufY[1][x];
			pixels2[x].rgbtRed = clipb[Y + cr];
         pixels2[x].rgbtGreen = clipb[Y + cg];
         pixels2[x].rgbtBlue = clipb[Y + cb];

  			Y = bufY[1][x+1];
			pixels2[x+1].rgbtRed = clipb[Y + cr];
         pixels2[x+1].rgbtGreen = clipb[Y + cg];
         pixels2[x+1].rgbtBlue = clipb[Y + cb];
		}

		drawPixels(0, y, pixels1, outwidth);
  		drawPixels(0, y+1, pixels2, outwidth);

		Fc  += 2*FcVF;
		src += 2*rawwidth;
	}

   unlockDisplay();

   if (show_settings_timeout > time(0))
   {
   	char buf[500];
	   sprintf(buf, "Brightness %d\n",
        brightness);
	   drawText(70, 120, buf, RGB(255, 255, 255));
   }
}
